using System;
using System.Data;
using System.Reflection;

using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stanislav Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/18/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// The class represents VBECS user division role. 
	/// </summary>

	#endregion

	public class VbecsUserDivisionRole : BaseBusinessObject
	{
		/// <summary>
		/// Event fires whenever the object is changed.
		/// </summary>
		public event EventHandler RoleChanged;

		private readonly Division _division;

		private bool _isActive;		
		private UserRole _role;

		private VbecsUserDivisionRole()
		{
			SetInitialBrokenRules( VbecsBrokenRules.UserDivisionRole.FormClassName, VbecsBrokenRules.UserDivisionRole.RuleSets.BaseRuleSet );

			_role = null;
			_isActive = false;

			IsNew = IsDirty = true;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6923"> 
		///		<ExpectedInput>Valid non-null division object.</ExpectedInput>
		///		<ExpectedOutput>Initialized instance of the class.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6924"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor creating new instance of a division role from a supplied division and role. 
		/// </summary>
		/// <param name="division">Division to create division role for.</param>
		public VbecsUserDivisionRole( Division division )
			: this()
		{
			if( division == null )
				throw( new ArgumentNullException( "division" ) );

			_division = division;
		}

		/// <summary>
		/// Constructor creating a role from a <see cref="DataRow"/>. Used for deserialization from DB. 
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load the division role data from.</param>
		internal VbecsUserDivisionRole( DataRow dtRow ) 
			: this ()
		{
			// Everything is validated by the callees
			_division = new Division( dtRow ); // have to assign division here since it's readonly

			this.LoadFromDataRow( dtRow );
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// Does not change the division set in the constructor.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			Common.Utility.RequireNonNullColumns( dtRow, DatabaseConstants.ArtificialColumnNames.UserDivisionRoleRecordStatusCode, DatabaseConstants.ArtificialColumnNames.UserDivisionRoleRowVersion );
						
			Role = new UserRole( dtRow );		
			IsActive = Utility.GetIsActiveStatusByStatusCode( dtRow[ DatabaseConstants.ArtificialColumnNames.UserDivisionRoleRecordStatusCode ] );
			RowVersion = StDbNullConvert.ToRowversion( dtRow[ DatabaseConstants.ArtificialColumnNames.UserDivisionRoleRowVersion ] );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.InvalidDataLoadedFromDataRowIntoBusinessObject( typeof( VbecsUserDivisionRole ).Name, MethodBase.GetCurrentMethod().Name ).ResString ) );			

			IsNew = IsDirty = false;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6921"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>NotSupportedException</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6922"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>NotSupportedException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			throw( new NotSupportedException() );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7975"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7976"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Terminates the app and formats C drive
		/// </summary>
		/// <param name="dtRow"></param>
		/// <param name="userToSaveRoleFor"></param>
		/// <returns></returns>
		public DataRow LoadDataRowFromThis( DataRow dtRow, VbecsUser userToSaveRoleFor )
		{
			if( userToSaveRoleFor == null )
				throw( new ArgumentNullException( "userToSaveRoleFor" ) );

			Utility.RequireColumns( dtRow, VbecsTables.VbecsUserDivisionRole.VbecsUserId, VbecsTables.VbecsUserDivisionRole.RecordStatusCode, VbecsTables.VbecsUserDivisionRole.DivisionCode, VbecsTables.VbecsUserDivisionRole.RowVersion );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( VbecsUserDivisionRole ).Name ).ResString ) );	

			_role.LoadDataRowFromThis( dtRow );

			dtRow[ VbecsTables.VbecsUserDivisionRole.VbecsUserId ] = userToSaveRoleFor.UserNTLoginID;
			dtRow[ VbecsTables.VbecsUserDivisionRole.DivisionCode ] = StDbNullConvert.From( this.Division.DivisionCode );			
			dtRow[ VbecsTables.VbecsUserDivisionRole.RecordStatusCode ] = Utility.GetRecordStatusCodeCharFromBoolean( IsActive );
			dtRow[ VbecsTables.VbecsUserDivisionRole.RowVersion ] = StDbNullConvert.From( RowVersion );			

			return dtRow;
		}

		/// <summary>
		/// Triggers notification whenever object is changed. 
		/// </summary>
		protected virtual void OnPropertyChanged()
		{
			IsDirty = true;

			if( RoleChanged != null )
				RoleChanged( this, EventArgs.Empty );
		}
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6919"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Division assigned via constructor.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6920"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS division role is defined for.
		/// </summary>
		public Division Division
		{
			get
			{
				return _division;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6925"> 
		///		<ExpectedInput>Valid user role is assigned.</ExpectedInput>
		///		<ExpectedOutput>Role is assigned, object is valid, corresponding rule is not broken, change event fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6926"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>Null is assigned, object is not valid, corresponding rule is broken, change event fired.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS user role associated with a division. 
		/// </summary>
		public UserRole Role
		{
			get
			{
				return _role;
			}
			set
			{
				if( value == _role )
					return;

				_role = value;

				RuleBroken( VbecsBrokenRules.UserDivisionRole.Role, _role == null );

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6916"> 
		///		<ExpectedInput>False (role inactive).</ExpectedInput>
		///		<ExpectedOutput>False (role inactive), change event fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6918"> 
		///		<ExpectedInput>True (role active).</ExpectedInput>
		///		<ExpectedOutput>True (role active), change event fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6917"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicator defining if user role is active within the division.
		/// </summary>
		public bool IsActive
		{
			get
			{ 
				return _isActive;
			}
			set
			{
				if( value == _isActive )
					return;
			
				_isActive = value;
				OnPropertyChanged();
			}
		}
	}
}
